(function () {
    const QUEUE_KEY = 'boonpos-offline-sales-queue';
    const OFFLINE_DEFAULT_MESSAGE = 'No network. Changes are cached locally and sync automatically when you reconnect.';
    const banner = document.getElementById('offline_sync_status');
    const messageEl = document.getElementById('offline_sync_message');
    const defaultMessage = (messageEl && messageEl.dataset.default) ? messageEl.dataset.default : OFFLINE_DEFAULT_MESSAGE;
    const form = document.getElementById('add_pos_sell_form');
    const csrfMeta = document.querySelector('meta[name="csrf-token"]');
    const csrfToken = csrfMeta ? csrfMeta.content : '';
    let storageAvailable = true;
    let isFlushing = false;

    try {
        const testKey = '__boonpos_storage_test__';
        localStorage.setItem(testKey, testKey);
        localStorage.removeItem(testKey);
    } catch (error) {
        storageAvailable = false;
    }

    function readQueue() {
        if (!storageAvailable) {
            return [];
        }
        try {
            const serialized = localStorage.getItem(QUEUE_KEY);
            return serialized ? JSON.parse(serialized) : [];
        } catch (error) {
            localStorage.removeItem(QUEUE_KEY);
            return [];
        }
    }

    function writeQueue(queue) {
        if (!storageAvailable) {
            return;
        }
        try {
            localStorage.setItem(QUEUE_KEY, JSON.stringify(queue));
        } catch (error) {
            console.warn('Unable to persist offline queue', error);
        }
    }

    function enqueue(payload) {
        if (!storageAvailable) {
            return;
        }
        const queue = readQueue();
        queue.push(payload);
        writeQueue(queue);
        updateBanner();
    }

    function serialize(formElement) {
        const data = [];
        if (!formElement) {
            return data;
        }
        const formData = new FormData(formElement);
        formData.forEach((value, key) => {
            data.push({ name: key, value });
        });
        return data;
    }

    function updateBanner() {
        if (!banner || !messageEl) {
            return;
        }
        const queue = readQueue();
        if (!navigator.onLine) {
            banner.style.display = 'block';
            const suffix = queue.length
                ? ` (${queue.length} sale${queue.length === 1 ? '' : 's'} queued)`
                : '';
            messageEl.textContent = `${defaultMessage}${suffix}.`;
            return;
        }

        if (queue.length) {
            banner.style.display = 'block';
            messageEl.textContent = `Synchronizing ${queue.length} pending sale${queue.length === 1 ? '' : 's'}.`;
            return;
        }

        banner.style.display = 'none';
    }

    async function sendQueueItem(item) {
        const body = new URLSearchParams();
        item.data.forEach((field) => {
            body.append(field.name, field.value);
        });

        const headers = {
            'X-Requested-With': 'XMLHttpRequest',
            Accept: 'application/json, text/plain, */*',
        };

        if (csrfToken) {
            headers['X-CSRF-TOKEN'] = csrfToken;
        }

        const response = await fetch(item.action, {
            method: item.method || 'POST',
            headers,
            credentials: 'same-origin',
            body,
        });

        if (!response.ok) {
            throw new Error(`Offline sync failed (${response.status})`);
        }
    }

    async function flushQueue() {
        if (!navigator.onLine || !storageAvailable || isFlushing) {
            return;
        }
        let queue = readQueue();
        if (!queue.length) {
            updateBanner();
            return;
        }
        isFlushing = true;
        while (queue.length) {
            const item = queue[0];
            try {
                await sendQueueItem(item);
                queue.shift();
                writeQueue(queue);
            } catch (error) {
                console.error('Offline queue sync error', error);
                break;
            }
        }
        isFlushing = false;
        updateBanner();
    }

    function handleFormSubmit(event) {
        if (navigator.onLine) {
            return;
        }
        event.preventDefault();
        const payload = {
            action: form.action,
            method: (form.getAttribute('method') || 'POST').toUpperCase(),
            data: serialize(form),
            createdAt: Date.now(),
        };
        enqueue(payload);
        if (window.toastr && typeof window.toastr.info === 'function') {
            window.toastr.info('Sale cached locally and will sync when online.', 'Offline mode');
        } else {
            console.info('Offline mode: your sale was cached and will sync automatically.');
        }
    }

    function registerServiceWorker() {
        if (!('serviceWorker' in navigator)) {
            return;
        }
        window.addEventListener('load', () => {
            navigator.serviceWorker
                .register('/sw.js')
                .catch((error) => console.warn('Service worker registration failed', error));
        });
    }

    document.addEventListener('DOMContentLoaded', () => {
        registerServiceWorker();
        updateBanner();
        flushQueue();
        if (form) {
            form.addEventListener('submit', handleFormSubmit);
        }
    });

    window.addEventListener('online', () => {
        updateBanner();
        flushQueue();
    });

    window.addEventListener('offline', updateBanner);

    setInterval(flushQueue, 10000);
})();
